--------------------------------------------------------------------------------
--  # SET ROLE is NOT allowed inside function (even not in called subfunctions)
--    which are designated with SECURITY DEFINER
--  # the same is true for SET SESSION AUTHORIZATION
--  # SET ROLE needs either SUPERUSER permissions OR
--    the role that wants to change to a different role, needs to be a member
--    of that role
--  # adding all login-roles to one 'master' -login-role is a security risk
--    all members could now switch to every other role that is a member
--    and thus gaining permissions of that role
--  # the solution is:
--     * use <GRANT [userrole] TO [masterloginrole]> (function) to allow putting the role that
--       we need to switch to into the 'master'-login-role
--     * now we can SET ROLE (to the role we just put as member inside ourself)
--     * remove the role we switched to from 'master'-login-role
--     * put this with authentication check inside a function
--       WITHOUT SESSION AUTHORIZATION and allow only the 'master'-login-role
--       EXECUTE permission
--------------------------------------------------------------------------------


---------------------------------------------------------------------------------
-- # never define functions for anything but
--   check_xyz     - secure check for delegated login method xyz
--   login_xyz     - secure login for delegated login method xyz
-- # only functions that need to be executed by master-login roles
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
CREATE OR REPLACE FUNCTION tsystem_security.delegatedlogin_masterlogin_grantrole(role_name text) RETURNS void AS $$
DECLARE
  _SQL varchar;

BEGIN
  -- must be session_user, the actually authenticated user that loged in to pg
  -- for security reasons
  _SQL = format('GRANT %I TO %I;', role_name, session_user);
  EXECUTE _SQL;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER;
-- \0050 System\000 base functions.sql\delegatedlogin_masterlogin_revokerole

--------------------------------------------------------------------------------
-- check bde-rfid login, without any provided user-role (returns the role found)
-- used for very simple nfc-login (just the nfc chip and its uid) when loged in as BDE
-- or as a masterlogin role
CREATE OR REPLACE FUNCTION tsystem_security.delegatedlogin_check_bde_rfid(rfid text) RETURNS varchar AS $$
DECLARE
  _count integer;
  _user  varchar;
  _ret   varchar;

BEGIN
  _ret = NULL;

  if (NOT(rfid IS NULL)) THEN
    SELECT
      count(*)
    FROM
      llv
    WHERE
      tsystem.enum_getvalue(ll_rfid, rfid::character varying, ','::character varying) -- list to check against, value to check, separator
      AND (ll_prszzeit OR ll_auftrzeit)
      AND COALESCE(ll_endd, current_date) <= current_date
    INTO
      _count
    ;

    IF (_count = 1) THEN
      SELECT
        ll_db_usename
      FROM
        llv
      WHERE
        tsystem.enum_getvalue(ll_rfid, rfid::character varying, ','::character varying) -- list to check against, value to check, separator
        AND (ll_prszzeit OR ll_auftrzeit)
        AND COALESCE(ll_endd, current_date) <= current_date
      INTO
        _user
      ;
    END IF;

    _ret = _user;
  END IF;

  RETURN _ret;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER;


--------------------------------------------------------------------------------
-- login with rfid, with provided user-role
-- used for very simple nfc-login (just the nfc chip and its uid) when loged in as a master-login role
CREATE OR REPLACE FUNCTION tsystem_security.delegatedlogin_login_rfid(role_name text, rfid text) RETURNS bool AS $$
DECLARE
  _ret boolean;

BEGIN
  SELECT
    tsystem.delegatedlogin_check_rfid(role_name, rfid)
  INTO
    _ret
  ;

  IF (_ret) THEN
    -- theese 3 commands should be ok without an exception block
    -- fail on 1. or run alll 3
    -- in worst case the 3. doesnt run, but then no exception block would help
    -- to mitigate problems from this last scenario
    --   add a check before these 3 commands, that no other role is member
    --   of session_user (master-login-role)
    --   then it will not allow 2 roles at same time within
    --   thus not giving chance to switch to another ordinary-login-role from
    --   ordinary-login-role
    PERFORM tsystem_security.delegatedlogin_masterlogin_grantrole(role_name);
    EXECUTE format('SET ROLE %I;', role_name);
    PERFORM tsystem.delegatedlogin_masterlogin_revokerole(role_name);
  END IF;

  RETURN _ret;
END;
$$ LANGUAGE plpgsql;